/*
 * Glib - Generic LIBrarian and editor
 *
 * Machine dependent stuff for MIDI programs.
 * Tim Thompson
 *
 * This is for MS-Windows 3.1, using Microsoft C/C++ 7.0.
 */

#include "glib.h"

#include <windows.h>
#include <mmsystem.h>
#include <commdlg.h>
#include <malloc.h>
#include <mmsystem.h>
#include <dos.h>
#include <time.h>
#include <io.h>
#include <direct.h>
#include <stdarg.h>

#include "glibdll.h"

/* MIDI status byte definitions */
#define NOTEON          0x90
#define NOTEOFF         0x80
#define KEYAFTERTOUCH   0xa0
#define CONTROLCHANGE   0xb0
#define PROGRAMCHANGE   0xc0
#define CHANAFTERTOUCH  0xd0
#define PITCHBEND       0xe0
#define SYSTEMMESSAGE   0xf0
#define BEGINSYSEX      0xf0
#define MTCQUARTERFRAME 0xf1
#define SONGPOSPTR      0xf2
#define SONGSELECT      0xf3

#define CIRCULAR_BUFF_SIZE 1000
#define MAX_NUM_DEVICES 8
#define NUM_OF_OUT_BUFFS 4
#define MIDI_OUT_BUFFSIZE 32
#define NUM_OF_IN_BUFFS 4
#define MIDI_IN_BUFFSIZE 8096
#define EBUFFSIZE 128

static HWND Khwnd;
static HDC Khdc;
static HANDLE Khinstance;
static HMIDIOUT hMidiOut;
static HMIDIIN hMidiIn[MAX_NUM_DEVICES];
static MIDIINCAPS midiInCaps[MAX_NUM_DEVICES];
static int wNumDevices = 0;
static LPCIRCULARBUFFER MidiInputBuffer;

int Nevents = 0;
int Nmidievents = 0;
int Nbytes = 0;

typedef struct mymidiinbuff {
	LPMIDIHDR midihdr;
} MY_MIDI_BUFF;

static MY_MIDI_BUFF *MidiInBuffs[MAX_NUM_DEVICES];
static MY_MIDI_BUFF *MidiOutBuffs;

LPCALLBACKINSTANCEDATA CallbackInputData[MAX_NUM_DEVICES];
LPCALLBACKINSTANCEDATA CallbackOutputData;

int Rows = 24;
int Cols = 80;

static int Curr_row = -1;
static int Curr_col = -1;
static int Fontwd, Fontht;
static int Maxx, Maxy;
static int Msx, Msy, Msb;
static long Lasttimeout;

long FAR PASCAL WndProc (HWND, UINT, UINT, LONG) ;
void freecursors(void);
int realmain(int,char **);
int windwaitfor(int);
int initmidi(void);
void endmidi(void);
int InitMidiInBuffs(int);
void freemidihdr(LPMIDIHDR);

#define K_CONSOLE 1
#define K_QUIT 2
#define K_TIMEOUT 3
#define K_MOUSE 4
#define K_MIDI 5
#define K_REDRAW 6

UINT Millires = 5;
UINT Milliwarn = 50;

void
logit(char *fmt,...)
{
	FILE *f = NULL;
	va_list args;

	f = fopen("debug.tmp","a");
	if ( f == NULL )
		windmsg("Hey, can't open debug.tmp!");
	else {
		va_start(args,fmt);
		vfprintf(f,fmt,args);
		fflush(f);
		fclose(f);
		va_end(args);
	}
}

void
windmsg(char *s)
{
	int n;
	n = MessageBox(Khwnd, s, "Glib", MB_ICONEXCLAMATION | MB_OKCANCEL);
	if ( n == IDCANCEL ) {
		bye();
		exit(0);
	}
}

static void
shiftup(int *arr,int n)
{
	int i;
	for ( i=n; i>0; i-- )
		arr[i] = arr[i-1];
}

#define MAXMOUSE 32
int Glibmousex[MAXMOUSE];
int Glibmousey[MAXMOUSE];
int Glibmouseb[MAXMOUSE];
int Glibmousepos = 0;

void
savemouse(x,y,b)
{
	if ( Glibmousepos >= MAXMOUSE )
		windmsg("Too many events received by savemouse!");
	else {
		shiftup(Glibmousex,Glibmousepos);
		shiftup(Glibmousey,Glibmousepos);
		shiftup(Glibmouseb,Glibmousepos);
		Glibmousex[0] = x;
		Glibmousey[0] = y;
		Glibmouseb[0] = b;
		Glibmousepos++;
	}
}

void
getmouseevent(void)
{
	if ( Glibmousepos <= 0 )
		windmsg("Hey, getmouseevent called when nothing to get!?");
	else {
		Msx = Glibmousex[--Glibmousepos];
		Msy = Glibmousey[Glibmousepos];
		Msb = Glibmouseb[Glibmousepos];
	}
}

#define MAXCONSOLE 32
int Glibconsole[MAXCONSOLE];
int Glibconsolepos = 0;

void
saveconsole(int n)
{
	/* silently throw away extra ones... */
	if ( Glibconsolepos < MAXCONSOLE ) {
		shiftup(Glibconsole,Glibconsolepos++);
		Glibconsole[0] = n;
	}
}

int
statconsole(void)
{
	return ( Glibconsolepos > 0 );
}

void
flushconsole(void)
{
	Glibconsolepos = 0;
}

int
getconsole(void)
{
	int c, r;

	while (1) {
		if ( Glibconsolepos <= 0 ) {
			int x, y;
			if ( Curr_row >= 0 ) {
				x = Curr_col*Fontwd;
				y = Curr_row*Fontht;
				SetROP2(Khdc,R2_BLACK);
				Rectangle(Khdc,x,y,x+Fontwd,y+Fontht);
			}
			r = windwaitfor(0);
			if ( r == K_QUIT )
				return -1;
			if ( r == K_REDRAW )
				return 0;
			if ( Curr_row >= 0 ) {
				SetROP2(Khdc,R2_WHITE);
				Rectangle(Khdc,x,y,x+Fontwd,y+Fontht);
			}
		}
		else {
			c = Glibconsole[--Glibconsolepos];
			break;
		}
	}
	return c;
}

#define MAXEVENTS 32
int Glibevents[MAXEVENTS];
int Glibeventpos = 0;

void
saveglibevent(int n)
{
	if ( Glibeventpos >= MAXEVENTS )
		windmsg("Too many events received by saveglibevent!");
	else {
		shiftup(Glibevents,Glibeventpos++);
		Glibevents[0] = n;
		if ( n == K_MIDI )
			Nmidievents++;
	}
}

int
getglibevent(void)
{
	if ( Glibeventpos <= 0 ) {
		return(-1);
	}
	else {
		int n = Glibevents[--Glibeventpos];
		switch (n) {
		case K_MOUSE:
			getmouseevent();
			break;
		case K_MIDI:
			Nmidievents--;
			break;
		}
		return n;
	}
}

#define IDM_ABOUT 100

int PASCAL WinMain (HANDLE hInstance, HANDLE hPrevInstance,
                    LPSTR lpszCmdParam, int nCmdShow)
{
	static char szAppName[] = "Glibnote" ;
	static char *argv[4];	/* static to avoid putting on stack */
	WNDCLASS    wndclass ;
	HMENU hmenu;
	DWORD dwrd;
	RECT rect;

	Khinstance = hInstance;
	if (!hPrevInstance) {
		wndclass.style         = CS_HREDRAW | CS_VREDRAW | CS_OWNDC ;
		wndclass.lpfnWndProc   = WndProc ;
		wndclass.cbClsExtra    = 0 ;
		wndclass.cbWndExtra    = 0 ;
		wndclass.hInstance     = hInstance ;
		wndclass.hIcon         = LoadIcon (Khinstance, "glib") ;
		wndclass.hCursor       = LoadCursor(0, IDC_ARROW);
		wndclass.hbrBackground = GetStockObject (WHITE_BRUSH) ;
		wndclass.lpszMenuName  = NULL ;
		wndclass.lpszClassName = szAppName ;
		
		RegisterClass (&wndclass) ;
	}

	dwrd = GetDialogBaseUnits();
	Fontwd = LOWORD(dwrd);
	Fontht = HIWORD(dwrd);

	rect.left = 0;
	rect.top = 0;
	rect.right = Fontwd * 81;	/* +1 for border */
	rect.bottom = Fontht * 26;	/* +2 for title */
	AdjustWindowRect( &rect, WS_OVERLAPPEDWINDOW, TRUE );

	Khwnd = CreateWindow (szAppName,         // window class name
		    "Glib",     // window caption
                    WS_OVERLAPPEDWINDOW,     // window style
                    CW_USEDEFAULT,           // initial x position
                    CW_USEDEFAULT,           // initial y position
                    rect.right,              // initial x size
                    rect.bottom,             // initial y size
                    (HWND) NULL,             // parent window handle
                    (HMENU) NULL,            // window menu handle
                    hInstance,               // program instance handle
		    NULL) ;		     // creation parameters

	ShowWindow (Khwnd, nCmdShow) ;
	UpdateWindow (Khwnd) ;
	Khdc = GetDC (Khwnd);

	hmenu = GetSystemMenu(Khwnd,FALSE);
	AppendMenu(hmenu,MF_SEPARATOR, 0, (LPSTR) NULL);
	AppendMenu(hmenu,MF_STRING, IDM_ABOUT, "About Glib...");

	argv[0] = "glib";
	argv[1] = lpszCmdParam;
	argv[2] = NULL;
	realmain(1,argv);

	return(0);
}

BOOL FAR PASCAL
AboutDlgProc (HWND hDlg, UINT message, UINT wParam, LONG lParam)
{
	switch(message) {
	case WM_INITDIALOG:
		return TRUE;
	case WM_COMMAND:
		switch(wParam) {
		case IDOK:
			EndDialog(hDlg, 0);
			return TRUE;
		case IDABORT:
			bye();
			exit(0);
		}
		break;
	}
	return FALSE;
}

long FAR PASCAL
WndProc (HWND hwnd, UINT message, UINT wParam, LONG lParam)
{
	int m;
	LPMIDIHDR midihdr;

	switch (message) {
	case WM_GLIB_TIMEOUT:
		saveglibevent(K_TIMEOUT);
		Lasttimeout = lParam;
		return 0;
	case WM_GLIB_ERROR:
		windmsg("WndProc got WM_GLIB_ERROR!");
		return 0;
	case WM_GLIB_MIDIINPUT:
		/* assumption is that when a midi event is seen, the code */
		/* will call getmidi() until all pending input is exhausted, */
		/* so we only have to make sure there's at least one K_MIDI */
		/* event pending. */
		if ( Nmidievents == 0 )
			saveglibevent(K_MIDI);
		return 0;
	case WM_GLIB_MIDIOUTPUT:
		midihdr = (LPMIDIHDR) lParam;
		midiOutUnprepareHeader(hMidiOut, midihdr,sizeof(MIDIHDR));
		freemidihdr(midihdr);
		return 0;
	case WM_CREATE:
		SelectObject (Khdc, GetStockObject (SYSTEM_FIXED_FONT)) ;
		return 0;
	case WM_SIZE:
		Maxx = LOWORD(lParam);
		Maxy = HIWORD(lParam);
		return 0;
	case WM_CHAR:
		saveglibevent(K_CONSOLE);
		saveconsole(wParam);
		return 0;
	case WM_SYSCOMMAND:
		switch (wParam) {
		case IDM_ABOUT: {
			FARPROC lpfnDlgProc = 
				MakeProcInstance (AboutDlgProc, Khinstance);
			DialogBox(Khinstance, "AboutBox", Khwnd, lpfnDlgProc);
			FreeProcInstance(lpfnDlgProc);
			}
			return 0;
		}
		break;

	case WM_LBUTTONDOWN:
	case WM_MBUTTONDOWN:
	case WM_RBUTTONDOWN:
	case WM_LBUTTONUP:
	case WM_MBUTTONUP:
	case WM_RBUTTONUP:
	case WM_MOUSEMOVE:
#ifdef OLDSTUFF
		if (message == WM_MOUSEMOVE )
			SetCursor(Kcursor);
#endif
		saveglibevent(K_MOUSE);
		m = 0;
		if ( wParam & MK_LBUTTON )
			m |= 1;
		else if ( (wParam & MK_RBUTTON) || (wParam & MK_MBUTTON) )
			m |= 2;
		savemouse(LOWORD(lParam),HIWORD(lParam),m);
		return 0;

	case WM_PAINT:
		Redraw = 1;
		saveglibevent(K_REDRAW);
		ValidateRect(hwnd, NULL);
		return 0 ;

          case WM_DESTROY:
		saveglibevent(K_QUIT);
               PostQuitMessage (0) ;
               return 0 ;
          }

	return DefWindowProc (hwnd, message, wParam, lParam) ;
}

void
bye(void)
{
	DestroyWindow(Khwnd);
	endmidi();
        PostQuitMessage (0) ;
	while ( windwaitfor(100) != K_QUIT )
		;
}

char *
alloc(n)
{
	char *p;

	if ( (p=malloc((unsigned)n)) == (char *)NULL ) {
		windmsg("*** Whoops *** alloc has failed?!?  No more memory!\n");
		bye();
	}
	return(p);
}

int
do1msg(void)
{
	MSG msg;

	/* usually, we get out because WM_DESTROY has done a */
	/* saveglibevent(K_QUIT), but just in case, we check */
	/* the return value of GetMessage anyway. */
	if ( GetMessage (&msg, (HWND) NULL, 0, 0) == 0 ) {
		return K_QUIT;
	}
	TranslateMessage (&msg) ;
	DispatchMessage (&msg) ;
	return 0;
}

int
windwaitfor(int tmout)
{
	int n;
	int timerid = 0;
	long tm0;

	if ( tmout >= 5 ) {
		tm0 = milliclock();
		timerid = timeSetEvent((UINT)tmout,(UINT)Millires,
			(LPTIMECALLBACK)GlibTimerFunc, 0L, TIME_ONESHOT);
		if ( timerid == 0 )
			windmsg("Unable to timeSetEvent!?");
	}
	while (1) {
		if ( (n=getglibevent()) > 0 )
			break;
		if ( do1msg() == K_QUIT )
			return K_QUIT;
	}
	if ( timerid != 0 ) {
		if ( n != K_TIMEOUT )
			timeKillEvent(timerid);
	}
	return n;
}

void
millisleep(int i)
{
	long stime = milliclock();
	long etime = stime + i;
	int r;

	while ( milliclock() < etime ) {
		r = windwaitfor((int)(etime-milliclock()));
		if ( r == K_QUIT )
			break;
	}

}

static unsigned char *bytes = NULL;
static int sofar;
static int nbytes;

int
statmidi(void)
{
	/* See if any queued up in low-level driver */
	if (MidiInputBuffer->dwCount > 0 )
		return 1;
	/* See if any saved up in buffer kept in getmidi() */
	if ( bytes != NULL && sofar < nbytes )
		return 1;
	/* Let events in. */
	(void) do1msg();
	return (MidiInputBuffer->dwCount > 0 );
}

/* getmidi reads data out of the circular midi buffer */
int
getmidi(void)
{
	static EVENT ev;
	static unsigned char smallbuff[3];
	static int waslong;
	LPMIDIHDR lpMidi;
	int r;

	if ( bytes == NULL ) {
		if ( GetEvent(MidiInputBuffer,&ev) == 0 )
			return -1;
		if ( ev.islong ) {
			lpMidi = (LPMIDIHDR)ev.data;
#ifdef OLDSTUFF
			char buff[140];
			sprintf(buff,"MIDIHDR leng=%ld recorded=%ld flags=0x%lx device=%ld",
				(long)(lpMidi->dwBufferLength),
				(long)(lpMidi->dwBytesRecorded),
				(long)(lpMidi->dwFlags),
				(long)(ev.dwDevice));
			windmsg(buff);
#endif
			if ( ((lpMidi->dwFlags) & MHDR_DONE) == 0 )
				windmsg("Hey, MHDR_DONE not set in dwFlags!?");
			bytes = lpMidi->lpData;
			nbytes = (int)(lpMidi->dwBytesRecorded);
			sofar = 0;
		}
		else {
			BYTE statusbyte;
			smallbuff[0] = LOBYTE(LOWORD(ev.data));
			statusbyte = smallbuff[0] & (BYTE)0xf0;
			switch ( statusbyte ) {

			/* Three byte events */
			case NOTEOFF:
			case NOTEON:
			case KEYAFTERTOUCH:
			case CONTROLCHANGE:
			case PITCHBEND:
				nbytes = 3;
				break;

			/* Two byte events */
			case PROGRAMCHANGE:
			case CHANAFTERTOUCH:
				nbytes = 2;
				break;

			/* MIDI system events (0xf0 - 0xff) */
			case SYSTEMMESSAGE:
			    switch(statusbyte) {
				/* Two byte system events */
				case MTCQUARTERFRAME:
				case SONGSELECT:
					nbytes = 2;
					break;
				/* Three byte system events */
				case SONGPOSPTR:
					nbytes = 3;
					break;
				/* One byte system events */
				default:
					nbytes = 1;
					break;
			    }
			    break;
            
			default:
				/* unknown event !? */
				nbytes = 0;
				break;
			}
			if ( nbytes > 1 ) {
				smallbuff[1] = HIBYTE(LOWORD(ev.data));
				if ( nbytes > 2 )
					smallbuff[2] = LOBYTE(HIWORD(ev.data));
			}
			bytes = smallbuff;
			sofar = 0;
		}
	}
	if ( bytes == NULL )
		r = -1;
	else {
		r = bytes[sofar++];
		if ( sofar >= nbytes ) {
			/* We've used up all the bytes */
			bytes = NULL;
			if ( ev.islong ) {
				/* give buffer back to the MIDI input device */
				int i = (int)(ev.dwDevice);
				if ( i < 0 || i >= wNumDevices ) {
					windmsg("Hey, bad ev.dwDevice!?");
				}
				else {
					midiInPrepareHeader(hMidiIn[i],
						(LPMIDIHDR)(ev.data),sizeof(MIDIHDR));
					midiInAddBuffer(hMidiIn[i],
						(LPMIDIHDR)(ev.data),sizeof(MIDIHDR));
				}
			}
		}
	}
	return r;
}

void
flushmidi(void)
{
	long eclk;

	while ( STATMIDI )
		getmidi();

	/* not sure if this is needed - just a stab at fixing a problem */
	eclk = milliclock() + 500;
	while ( milliclock() < eclk )
		;
}

/* getmousepos - get currect row and column of mouse */
void
getmousepos(amr,amc)
int *amr;
int *amc;
{
	*amr = -1;
	*amc = -1;
}

/* statmouse - return mouse button state (0=nothing pressed,1=left,2=right) */
int
statmouse(void)
{
	return(-1);
}

/* Return when either a console key or mouse button is pressed. */
int
mouseorkey(void)
{
	return(getconsole());
}

void
windinit(void)
{
}

void
windgoto(r,c)
int r,c;
{
	Curr_row = r;
	Curr_col = c; 
}

void
winderaserow(r)
int r;
{
	SetROP2(Khdc,R2_WHITE);
	Rectangle(Khdc,0,r*Fontht,Maxx,(r+1)*Fontht);
	SetROP2(Khdc,R2_BLACK);
}

void
windrefresh(void)
{
}

void
windclear(void)
{
	SetROP2(Khdc,R2_WHITE);
	Rectangle(Khdc,0,0,Maxx,Maxy);
	SetROP2(Khdc,R2_BLACK);
	windgoto(-1,-1);
}

void
windputc(c)
int c;
{
	char s[2];
	int x = Curr_col*Fontwd;
	int y = Curr_row*Fontht;

	switch(c) {
	case '|':
		MoveToEx(Khdc,x+Fontwd/2,y,(LPPOINT)NULL);
		LineTo(Khdc,x+Fontwd/2,y+Fontht);
		break;
	case '=':
		MoveToEx(Khdc,x,y+Fontht/3,(LPPOINT)NULL);
		LineTo(Khdc,x+Fontwd,y+Fontht/3);
		MoveToEx(Khdc,x,y+2*Fontht/3,(LPPOINT)NULL);
		LineTo(Khdc,x+Fontwd,y+2*Fontht/3);
		break;
	case '-':
		MoveToEx(Khdc,x,y+Fontht/2,(LPPOINT)NULL);
		LineTo(Khdc,x+Fontwd,y+Fontht/2);
		break;
	case '\b':
		Curr_col -= 2;
		break;
	case ' ':
		SetROP2(Khdc,R2_WHITE);
		Rectangle(Khdc,x,y,x+Fontwd,y+Fontht);
		SetROP2(Khdc,R2_BLACK);
		break;
	default:
		s[0] = c;
		s[1] = '\0';
		SelectObject (Khdc, GetStockObject (SYSTEM_FIXED_FONT)) ;
		TextOut(Khdc,x,y,s,1);
		break;
	}
	Curr_col++;
	windgoto(Curr_row,Curr_col);
}

/* windgets - get a line of input from the console, handling backspaces */
void
windgets(s)
char *s;
{
	char *origs = s;
	int c;

	while ( (c=getconsole()) != '\n' && c!='\r' && c!= EOF ) {
		if ( c == '\b' ) {
			if ( s > origs ) {
				windstr("\b \b");
				s--;
			}
		}
		else {
			windputc(c);
			*s++ = c;
		}
		windrefresh();
	}
	*s = '\0';
}

void
windstr(s)
char *s;
{
	while ( s!=NULL && *s != '\0' )
		windputc(*s++);
}

void
beep(void)
{
	MessageBeep(0);
}

void
windhigh(void)
{
}

void
windnorm(void)
{
}

/****************
 * openls(), nextls(), and closels() are used to scan the current directory.
 ***************/

static int first = 1;

void
openls(void)
{
	first = 1;
}

char *
nextls(void)
{
	WIN32_FIND_DATA fd;
	static HANDLE h;

	if ( first ) {
		h = FindFirstFile("*.*",&fd);
		first = 0;
	}
	else {
		if ( FindNextFile(h,&fd) == FALSE )
			h = INVALID_HANDLE_VALUE;
	}

	if ( h == INVALID_HANDLE_VALUE )
		return((char *)NULL);
	else
		return(fd.cFileName);
}

void
closels(void)
{
}

void
hello(void)
{
	initmidi();
}

void
rtend(void)
{
}

static int Minit = 0;

int
initmidi(void)
{
	TIMECAPS tc;
	int r, i;
	char ebuff[EBUFFSIZE];

	timeGetDevCaps(&tc, sizeof(TIMECAPS));
	if ( tc.wPeriodMin > (UINT)Millires ) {
		char buff[128];
		Millires = tc.wPeriodMin;
		if ( Milliwarn < Millires ) {
			sprintf(buff,"Warning - wPeriodMin=%d Millires=%ld - Realtime may suffer",
				tc.wPeriodMin,Millires);
			windmsg(buff);
		}
	}

	if ( timeBeginPeriod((UINT)Millires) ) {
		windmsg("timeBeginPeriod fail!?");
		return 1;
	}

	if ( GlibSetupMidi(Khwnd) ) {
		windmsg("GlibSetupMidi fail!?");
		return 1;
	}

	/* Get the number of MIDI input devices.  Then get the capabilities of
	 * each device.  We don't use the capabilities information right now,
	 * but we could use it to report the name of the device that received
	 * each MIDI event.
	 */
	wNumDevices = midiInGetNumDevs();
	if (!wNumDevices) {
		windmsg("There are no MIDI input devices.");
		return 1;
	}
	for (i=0; (i<wNumDevices) && (i<MAX_NUM_DEVICES); i++) {
		r = midiInGetDevCaps(i, (LPMIDIINCAPS) &midiInCaps[i],
                                sizeof(MIDIINCAPS));
		if(r) {
			midiInGetErrorText(r, ebuff, EBUFFSIZE);
			windmsg(ebuff);
		}
	}

	/* Allocate a circular buffer for low-level MIDI input.  This buffer
	 * is filled by the low-level callback function and emptied by the
	 * application when it receives MM_MIDIINPUT messages.
	 */
	MidiInputBuffer = AllocCircularBuffer((DWORD)(CIRCULAR_BUFF_SIZE));
	if (MidiInputBuffer == NULL) {
		windmsg("Not enough memory available for input buffer.");
		return 1;
	}

	/* Open all MIDI input devices after allocating and setting up
	 * instance data for each device.  The instance data is used to
	 * pass buffer management information between the application and
	 * the low-level callback function.  A single callback function 
	 * is used to service all opened input devices.
	 */
	for (i=0; (i<wNumDevices) && (i<MAX_NUM_DEVICES); i++) {
		if ((CallbackInputData[i] = AllocCallbackInstanceData()) == NULL) {
			windmsg("Not enough memory available.");
			FreeCircularBuffer(MidiInputBuffer);
			return 1;
		}
		CallbackInputData[i]->hWnd = Khwnd;         
		CallbackInputData[i]->dwDevice = i;
		CallbackInputData[i]->lpBuf = MidiInputBuffer;

		r = midiInOpen((LPHMIDIIN)&hMidiIn[i],
				  i,
				  (DWORD)midiInputHandler,
				  (DWORD)CallbackInputData[i],
				  CALLBACK_FUNCTION);
		if(r) {
			/* can't open */
			FreeCallbackInstanceData(CallbackInputData[i]);
			midiInGetErrorText(r, ebuff, EBUFFSIZE);
			windmsg(ebuff);
		}
		else {
			/* successful open */
			if ( InitMidiInBuffs(i) ) {
				midiInStart(hMidiIn[i]);
			}
			else {
				windmsg("Unable to allocate Midi Buffers!?");
			}
		}
	}
	
	if ((CallbackOutputData = AllocCallbackInstanceData()) == NULL) {
		windmsg("Not enough memory available!?");
		return 1;
	}
	CallbackOutputData->hWnd = Khwnd;         
	r = midiOutOpen ( &hMidiOut, MIDI_MAPPER,
			(DWORD)midiOutputHandler,
			(DWORD)CallbackOutputData,
			(DWORD) CALLBACK_FUNCTION);
	if ( r ) {
		midiOutGetErrorText(r,ebuff,EBUFFSIZE);
		windmsg(ebuff);
		return 1;
	}
	/* InitMidiOutBuffs(); */

	Minit = 1;
	return 0;
}

void
endmidi(void)
{
	if ( Minit ) {
		int i;

		timeEndPeriod((UINT)Millires);

		/* HEY!! midiOutReset() sends so much data so quickly */
		/* that it overflows the buffers my TX81Z and MX8 (at least, */
		/* when running on my 486/50 machine) - this seem rude. */
		/* Hence this is commented out. */

		/* midiOutReset(hMidiOut); */

		/* HEY!! midiOutClose() does the same thing!!!  I don't */
		/* think we should avoid calling midiOutClose, though. */

		midiOutClose(hMidiOut);

		for (i=0; (i<wNumDevices) && (i<MAX_NUM_DEVICES); i++) {
			if ( hMidiIn[i] == 0 )
				continue;
			midiInStop(hMidiIn[i]);
			midiInReset(hMidiIn[i]);
			midiInClose(hMidiIn[i]);
			/* NEED TO FREE GLOBAL BUFFERS AND STUFF!!! */
		}
	}
}

long
milliclock(void)
{
	return (long)timeGetTime();
}
void
resetclock(void)
{
}

/* filetime - Return the modification time of a file in seconds. */
long
filetime(fn)
char *fn;	/* file name */
{
	struct stat s;

	if ( stat(fn,&s) == -1 )
		return(-1);
	return(s.st_mtime);
}

/* currtime - Return current time in seconds (consistent with filetime()) */
long
currtime(void)
{
	return ( time((long *)0) );
}

LPMIDIHDR
newmidihdr(int sz)
{
	LPMIDIHDR lpMidi;
	HPSTR lpData;

	/* Allocate and lock the buffer .  */
	lpData = (HPSTR) GlobalAlloc(0, (DWORD)sz);
	if(lpData == NULL)
		return 0;
	    
	/* Allocate and lock a MIDIHDR structure.  */
	lpMidi = (LPMIDIHDR) GlobalAlloc(0, (DWORD)sizeof(MIDIHDR));
	if(lpMidi == NULL)
		return 0;
	    
	lpMidi->lpData = lpData;
	lpMidi->dwBufferLength = sz;
	lpMidi->dwBytesRecorded = 0;
	lpMidi->dwUser = 0;
	lpMidi->dwFlags = 0;
	return lpMidi;
}

void
freemidihdr(LPMIDIHDR midihdr)
{
	GlobalFree(midihdr->lpData);
	GlobalFree(midihdr);
}

/*
 * InitMidiInBuffs - Allocates MIDIHDR structures and buffers for port i.
 */

int
InitMidiInBuffs(int i)
{
	int n;

	MidiInBuffs[i] = malloc( NUM_OF_IN_BUFFS * sizeof (MY_MIDI_BUFF) );

	for ( n=0; n<NUM_OF_IN_BUFFS; n++ ) {
	
		MidiInBuffs[i][n].midihdr = newmidihdr(MIDI_IN_BUFFSIZE);

		midiInPrepareHeader(hMidiIn[i],
			MidiInBuffs[i][n].midihdr,sizeof(MIDIHDR));
		midiInAddBuffer(hMidiIn[i],
			MidiInBuffs[i][n].midihdr,sizeof(MIDIHDR));
	}
	return 1;
}

void
putnmidi(int n,BYTE *p)
{
	LPMIDIHDR midihdr;
	BYTE *q;
	int r;
	char ebuff[EBUFFSIZE];

#ifdef NEWSTUFF
	if ( n == 3 ) {
		union {
			DWORD dwData;
			BYTE bData[4];
		} u;
		u.bData[0] = p[0];
		u.bData[1] = p[1];
		u.bData[2] = p[2];
		u.bData[3] = 0;
		r = midiOutShortMsg(hMidiOut, u.dwData);
		if ( r ) {
			windmsg("Error in midiOutShortMsg (description follows)");
			midiOutGetErrorText(r,ebuff,EBUFFSIZE);
			windmsg(ebuff);
			return;
		}
	}
	else 
#endif
	{
		midihdr = newmidihdr(n);
		if ( midihdr == 0 ) {
			windmsg("Unable to allocate midihdr in putnmidi!?");
			return;
		}
		r = midiOutPrepareHeader(hMidiOut, midihdr,sizeof(MIDIHDR));
		if ( r ) {
			windmsg("Error in midiOutPrepareHeader (description follows)");
			midiOutGetErrorText(r,ebuff,EBUFFSIZE);
			windmsg(ebuff);
			return;
		}
		for ( q=midihdr->lpData; n-- > 0; ) {
			*q++ = *p++;
		}
		r = midiOutLongMsg(hMidiOut,midihdr,sizeof(MIDIHDR));
		if ( r ) {
			windmsg("Error in midiOutLongMsg (description follows)");
			midiOutGetErrorText(r,ebuff,EBUFFSIZE);
			windmsg(ebuff);
			return;
		}
	}
}

int sendsofar = 0;
#define SENDSIZE 128
BYTE sendbuff[SENDSIZE];

void
sendflush(void)
{
	if ( sendsofar > 0 ) {
		putnmidi(sendsofar,sendbuff);
		sendsofar = 0;
	}
}

void
sendmidi(int val)
{
	sendbuff[sendsofar++] = val;
	if ( val == EOX || sendsofar >= SENDSIZE )
		sendflush();
}

/*
 * circbuf.c - Routines to manage the circular MIDI input buffer.
 *      This buffer is filled by the low-level callback function and
 *      emptied by the application.  Since this buffer is accessed
 *      by a low-level callback, memory for it must be allocated
 *      exactly as shown in AllocCircularBuffer().
 */

/*
 * AllocCircularBuffer -    Allocates memory for a CIRCULARBUFFER structure 
 * and a buffer of the specified size.
 *
 * Params:  dwSize - The size of the buffer, in events.
 *
 * Return:  A pointer to a CIRCULARBUFFER structure identifying the 
 *      allocated display buffer.  NULL if the buffer could not be allocated.
 */
LPCIRCULARBUFFER
AllocCircularBuffer(DWORD dwSize)
{
    LPCIRCULARBUFFER lpBuf;
    LPEVENT lpMem;
    
    lpBuf = (LPCIRCULARBUFFER) GlobalAlloc(0, (DWORD)sizeof(CIRCULARBUFFER));
    if(lpBuf == (LPCIRCULARBUFFER) NULL)
        return NULL;
    
    lpMem = (LPEVENT) GlobalAlloc(0, dwSize * sizeof(EVENT));
    if(lpMem == NULL)
        return NULL;
    
    lpBuf->hBuffer = lpMem;
    lpBuf->wError = 0;
    lpBuf->dwSize = dwSize;
    lpBuf->dwCount = 0L;
    lpBuf->lpStart = lpMem;
    lpBuf->lpEnd = lpMem + dwSize;
    lpBuf->lpTail = lpMem;
    lpBuf->lpHead = lpMem;
        
    return lpBuf;
}

/* FreeCircularBuffer - Frees the memory for the given CIRCULARBUFFER 
 * structure and the memory for the buffer it references.
 *
 * Params:  lpBuf - Points to the CIRCULARBUFFER to be freed.
 *
 * Return:  void
 */
void FreeCircularBuffer(LPCIRCULARBUFFER lpBuf)
{
    GlobalFree(lpBuf->hBuffer);
    GlobalFree(lpBuf);
}

/* GetEvent - Gets a MIDI event from the circular input buffer.  Events
 *  are removed from the buffer.  The corresponding PutEvent() function
 *  is called by the low-level callback function, so it must reside in
 *  the callback DLL.  PutEvent() is defined in the CALLBACK.C module.
 *
 * Params:  lpBuf - Points to the circular buffer.
 *          lpEvent - Points to an EVENT structure that is filled with the
 *              retrieved event.
 *
 * Return:  Returns non-zero if successful, zero if there are no 
 *   events to get.
 */
WORD FAR PASCAL
GetEvent(LPCIRCULARBUFFER lpBuf, LPEVENT lpEvent)
{
    /* If no event available, return.
     */
    if(lpBuf->dwCount <= 0)
        return 0;
    
    /* Get the event.
     */
    *lpEvent = *lpBuf->lpTail;
    
    /* Decrement the byte count, bump the tail pointer.
     */
    --lpBuf->dwCount;
    ++lpBuf->lpTail;
    
    /* Wrap the tail pointer, if necessary.
     */
    if(lpBuf->lpTail >= lpBuf->lpEnd)
        lpBuf->lpTail = lpBuf->lpStart;

    return 1;
}

/* AllocCallbackInstanceData - Allocates a CALLBACKINSTANCEDATA
 *      structure.  This structure is used to pass information to the
 *      low-level callback function, each time it receives a message.
 *
 * Params:  void
 *
 * Return:  A pointer to the allocated CALLBACKINSTANCE data structure.
 */
LPCALLBACKINSTANCEDATA FAR PASCAL AllocCallbackInstanceData(void)
{
    LPCALLBACKINSTANCEDATA lpBuf;
    
    lpBuf = (LPCALLBACKINSTANCEDATA) GlobalAlloc(0,(DWORD)sizeof(CALLBACKINSTANCEDATA));
    if(lpBuf == NULL)
        return NULL;
    return lpBuf;
}

/* FreeCallbackInstanceData - Frees the given CALLBACKINSTANCEDATA structure.
 *
 * Params:  lpBuf - Points to the CALLBACKINSTANCEDATA structure to be freed.
 *
 * Return:  void
 */
void FAR PASCAL FreeCallbackInstanceData(LPCALLBACKINSTANCEDATA lpBuf)
{
    GlobalFree(lpBuf);
}

char *
myeventinfo(void)
{
	static char buff[100];
	sprintf(buff,"(Nbytes=%d Cnt=%ld Size=%ld  Err=%d)",
		Nbytes,
		(long) (MidiInputBuffer->dwCount),
		(long) (MidiInputBuffer->dwSize),
		MidiInputBuffer->wError);
	return(buff);
}
